#include <vector>
#include <stdlib.h>
#include <iostream>
#include <cmath>
#include <cstring>
#include <algorithm>
#include <fstream>
#include <memory>
#include <random>
#include <malloc.h>
#include <sstream>
#include <iostream>
#include <fstream>
#include <thread>
#include <mutex>
#include <vector>

class EdgeV1;
class VertexV1;

class VertexV1 {
    public: 

    int part = 0; //part = 0 means left, part = 1 means right 
    int id = 0;
    std::vector<EdgeV1*> edges = std::vector<EdgeV1*>();
    bool isRemained = true;
    bool isFrozen = false; 
    int sampledDegree = 0; 
    
    VertexV1(int id=0, int part=0) : id(id), part(part) {}
    VertexV1(int id, std::vector<EdgeV1*> edges) : id(id), edges(edges) {}
};

class EdgeV1 {
    public:

    VertexV1* v1 = nullptr;
    VertexV1* v2 = nullptr;
    int id = 0; 

    bool isRemained = true; 
    int leftId = 0;
    int rightId = 0;
    int edgeListId = 0;

    EdgeV1(VertexV1* v1, VertexV1* v2, int id=0) : v1(v1), v2(v2), id(id) {}
};

class GraphV1{
    public: 
    std::vector<std::unique_ptr<VertexV1>> leftVertices=std::vector<std::unique_ptr<VertexV1>>(), rightVertices=std::vector<std::unique_ptr<VertexV1>>();
    std::vector<std::unique_ptr<EdgeV1>> edges = std::vector<std::unique_ptr<EdgeV1>>();

    std::vector<std::unique_ptr<VertexV1>> remainedLeftVertices=std::vector<std::unique_ptr<VertexV1>>(), remainedRightVertices=std::vector<std::unique_ptr<VertexV1>>();
    std::vector<std::unique_ptr<EdgeV1>> remainedEdges = std::vector<std::unique_ptr<EdgeV1>>();;
    std::vector<VertexV1*> frozenLeftVertices=std::vector<VertexV1*>(), frozenRightVertices=std::vector<VertexV1*>();
    
    int n = 0; 

    GraphV1(std::vector<std::pair<int, int>*> edges, int n){
        this->n = n;
        for(int i = 0; i < n; i++){
            auto v = std::make_unique<VertexV1>(i, 0);
            leftVertices.push_back(std::move(v));
        }
        for(int i = 0; i < n; i++){
            auto v = std::make_unique<VertexV1>(i, 1);
            rightVertices.push_back(std::move(v));
        }
        for (int i = 0; i < edges.size(); i++) {
            auto e = std::make_unique<EdgeV1>(leftVertices[edges[i]->first].get(), 
                                              rightVertices[edges[i]->second].get(), i);
            
            leftVertices[edges[i]->first]->edges.push_back(e.get());
            rightVertices[edges[i]->second]->edges.push_back(e.get());
            this->edges.push_back(std::move(e));
        }
    }

    void initialize_remained_graph(){
        for(int i = 0; i < leftVertices.size(); i++){
            auto v = std::make_unique<VertexV1>(i, 0);
            remainedLeftVertices.push_back(std::move(v));
        }
        for(int i = 0; i < rightVertices.size(); i++){
            auto v = std::make_unique<VertexV1>(i, 1);
            remainedRightVertices.push_back(std::move(v));
        }
        for (int i = 0; i < edges.size(); i++) {
            auto e = std::make_unique<EdgeV1>(remainedLeftVertices[edges[i]->v1->id].get(),
                                              remainedRightVertices[edges[i]->v2->id].get(), i);
            
            e->leftId = remainedLeftVertices[edges[i]->v1->id]->edges.size();
            remainedLeftVertices[edges[i]->v1->id]->edges.push_back(e.get());

            e->rightId = remainedRightVertices[edges[i]->v2->id]->edges.size();
            remainedRightVertices[edges[i]->v2->id]->edges.push_back(e.get());
            
            e->edgeListId = this->remainedEdges.size();
            this->remainedEdges.push_back(std::move(e)); 
        }
        
    }

    void freeze_degree_gre(int leftThreshold, int rightThreshold){
        for(int i = 0; i < remainedLeftVertices.size(); i++){
            if(remainedLeftVertices[i]->edges.size() >= leftThreshold){
                remainedLeftVertices[i]->isFrozen = true;
                frozenLeftVertices.push_back(remainedLeftVertices[i].get());
            }
        }
        for(int i = 0; i < remainedRightVertices.size(); i++){
            if(remainedRightVertices[i]->edges.size() >= rightThreshold){
                remainedRightVertices[i]->isFrozen = true;
                frozenRightVertices.push_back(remainedRightVertices[i].get());
            }
        }
    }

    void get_degrees_in_sampled_graph(long double leftP, long double rightP){
        for(int i = 0; i < remainedEdges.size(); i++){
            if(remainedEdges[i]->v1->isRemained && remainedEdges[i]->v2->isRemained && !remainedEdges[i]->v1->isFrozen && !remainedEdges[i]->v2->isFrozen){
                if(1.L*rand() / RAND_MAX <= leftP){ //todo: check if this is correct
                    remainedEdges[i]->v1->sampledDegree++;
                }
                if(1.L*rand() / RAND_MAX <= rightP){ //todo: check if this is correct
                    remainedEdges[i]->v2->sampledDegree++;
                }
            }
        }
    }

    void get_left_degrees_in_sampled_graph(long double leftP){
        static std::random_device rd;
        static std::mt19937 gen(rd());
        static std::uniform_real_distribution<long double> dis(0.0, 1.0);

        for(int i = 0; i < remainedEdges.size(); i++){
            if(remainedEdges[i]->v1->isRemained && remainedEdges[i]->v2->isRemained && !remainedEdges[i]->v1->isFrozen && !remainedEdges[i]->v2->isFrozen){
                if(dis(gen) <= leftP){
                    remainedEdges[i]->v1->sampledDegree++;
                }
            }
        }
    }

    void get_right_degrees_in_sampled_graph(long double rightP){
        static std::random_device rd;
        static std::mt19937 gen(rd());
        static std::uniform_real_distribution<long double> dis(0.0, 1.0);

        for(int i = 0; i < remainedEdges.size(); i++){
            if(remainedEdges[i]->v1->isRemained && remainedEdges[i]->v2->isRemained && !remainedEdges[i]->v1->isFrozen && !remainedEdges[i]->v2->isFrozen){
                if(dis(gen) <= rightP){
                    remainedEdges[i]->v2->sampledDegree++;
                }
            }
        }
    }

    void remove_left_vertex(int leftVertexId){
        VertexV1* v = remainedLeftVertices[leftVertexId].get();
        std::swap(remainedLeftVertices[leftVertexId], remainedLeftVertices.back());
        for(int j = remainedLeftVertices.back()->edges.size() - 1; j >= 0; j--){
            EdgeV1 *e = remainedLeftVertices.back()->edges[j];
            remove_edge(e);
        }
        remainedLeftVertices.pop_back();
    }

    void remove_right_vertex(int rightVertexId){
        VertexV1* v = remainedRightVertices[rightVertexId].get();
        std::swap(remainedRightVertices[rightVertexId], remainedRightVertices.back());
        for(int j = remainedRightVertices.back()->edges.size() - 1; j >= 0; j--){
            EdgeV1 *e = remainedRightVertices.back()->edges[j];
            remove_edge(e);
        }
        remainedRightVertices.pop_back();
    }

    void remove_edge(EdgeV1* e){
        int rightId = e->rightId;
        int leftId = e->leftId;
        int edgeListId = e->edgeListId;

        std::swap(e->v1->edges[leftId]->leftId, e->v1->edges.back()->leftId);
        std::swap(e->v1->edges[leftId], e->v1->edges.back());

        std::swap(e->v2->edges[rightId]->rightId, e->v2->edges.back()->rightId);
        std::swap(e->v2->edges[rightId], e->v2->edges.back());

        std::swap(remainedEdges[edgeListId]->edgeListId, remainedEdges.back()->edgeListId);
        std::swap(remainedEdges[edgeListId], remainedEdges.back());

        e->v1->edges.pop_back();
        e->v2->edges.pop_back();
        remainedEdges.pop_back();
    }

    long double get_density(){
        int non_frozen_edges = 0; 
        for(int i = 0; i < remainedEdges.size(); i++){
            if(!remainedEdges[i]->v1->isFrozen && !remainedEdges[i]->v2->isFrozen)
                non_frozen_edges++;
        }

        long double first_candidate = 0; 
        if(remainedLeftVertices.size() > 0 && remainedRightVertices.size() > 0)
        first_candidate = 1.L*remainedEdges.size()/sqrtl((1.L*remainedLeftVertices.size()*remainedRightVertices.size()));

        long double second_candidate = 0.L; 
        if(remainedLeftVertices.size() - frozenLeftVertices.size() > 0 && remainedRightVertices.size() - frozenRightVertices.size() > 0 )
        second_candidate = 1.L*non_frozen_edges/sqrtl(((1.L*remainedLeftVertices.size() - frozenLeftVertices.size())*(remainedRightVertices.size() - frozenRightVertices.size())));
        return std::max(first_candidate, second_candidate);
    }
};

void printMemoryUsage() {
    struct mallinfo2 mi = mallinfo2();
    std::cout << "used: " << mi.uordblks << " free:" << mi.fordblks << std::endl;
}

class OurMPCAlgorithm{
    public:
    long double t1, t2; //thresholds for freezing edges
    long double k1, k2; //degree thresholds
    long double p1, p2; //probabilities for sampling 
    long double delta, epsilon; 
    long double candidate_z, candidate_d; 
    int T; //number of steps
    int R; //number of phases
    int number_of_rounds = 0; 
    std::string test_case;
    std::string algorithm_name = "OurMPCAlgorithm";
    GraphV1* G;
    

    OurMPCAlgorithm(GraphV1* G, long double delta, long double epsilon, std::string test_case = "test"){
        this->G = G;
        this->delta = delta;
        this->epsilon = epsilon;
        this->test_case = test_case;
        std::cout << "OurMPCAlgorithm initialized with delta: " << delta << ", epsilon: " << epsilon << ", test_case: " << test_case << std::endl;
    }
    
    void set_up_params_v1(long double candidate_d, long double candidate_z){
        this->candidate_d = candidate_d;
        this->candidate_z = candidate_z;
        this->k1 = candidate_d / (2.L*candidate_z);
        this->k2 = candidate_d*candidate_z / (2.L);
        this->p1 = std::min(1.L, 18.L * logl(G->n) / (epsilon * epsilon * k1));
        this->p2 = std::min(1.L, 18.L * logl(G->n) / (epsilon * epsilon * k2));
        this->t1 = k1 * powl((1+epsilon), sqrtl(logl(G->n)*delta));
        this->t2 = k2 * powl((1+epsilon), sqrtl(logl(G->n)*delta)); 
        this->T = std::ceil(sqrtl(logl(G->n)*delta)/2.L);
        this->R = std::ceil(8*sqrtl(logl(G->n)*delta)/delta);
        this->G->initialize_remained_graph();
    }
    
    void finish_left_step(){
        for(int i = 0; i < G->remainedLeftVertices.size(); i++){
            if(!G->remainedLeftVertices[i]->isFrozen && G->remainedLeftVertices[i]->sampledDegree < p1*k1){
                G->remove_left_vertex(i);
                i--;
            }
            else
                G->remainedLeftVertices[i]->sampledDegree = 0;
        }
    }

    void finish_right_step(){
        for(int i = 0; i < G->remainedRightVertices.size(); i++){
            if(!G->remainedRightVertices[i]->isFrozen && G->remainedRightVertices[i]->sampledDegree < p2*k2){
                G->remove_right_vertex(i);
                i--;
            }
            else 
                G->remainedRightVertices[i]->sampledDegree = 0;
        }
    }
    
    

    void finish_phase(){
        for(int i= 0 ; i < G->frozenLeftVertices.size(); i++){
            G->frozenLeftVertices[i]->isFrozen= 0;
        }
        for(int i= 0 ; i < G->frozenRightVertices.size(); i++){
            G->frozenRightVertices[i]->isFrozen = 0;
        }
        G->frozenLeftVertices.clear();
        G->frozenRightVertices.clear();
    }

    void free_remained(){
        G->frozenLeftVertices.clear();
        G->frozenRightVertices.clear();
        G->remainedLeftVertices.clear();
        G->remainedRightVertices.clear();
        G->remainedEdges.clear();
    }

    std::pair<int, long double>  end_run_for_fixed_z_d(){
        std::pair<int, long double> ret = {number_of_rounds, G->get_density()};
        // std::cout << "remainedLeftVertices: " << G->remainedLeftVertices.size() << "   remainedRightVertices: " << G->remainedRightVertices.size() << std::endl;
        // std::cout << "frozenLeftVertices: " << G->frozenLeftVertices.size() << "   frozenRightVertices: " << G->frozenRightVertices.size() << std::endl;
        free_remained();
        return ret; 
    }

    void print_params(){
        std::cout << "candidate_d: " << candidate_d << std::endl;
        std::cout << "candidate_z: " << candidate_z << std::endl;
        std::cout << "k1: " << k1 << std::endl;
        std::cout << "k2: " << k2 << std::endl;
        std::cout << "p1: " << p1 << std::endl;
        std::cout << "p2: " << p2 << std::endl;
        std::cout << "t1: " << t1 << std::endl;
        std::cout << "t2: " << t2 << std::endl;
        std::cout << "T: " << T << std::endl;
        std::cout << "R: " << R << std::endl;
    }
    
    std::pair<int, long double> run_for_fixed_z_d(long double candidate_z, long double candidate_d){
        this->set_up_params_v1(candidate_d, candidate_z);
        // print_params();
        number_of_rounds = 0;

        for(int phase_number = 0; phase_number < R; phase_number++){
            long double f1_treshold = candidate_z*sqrtl(G->remainedLeftVertices.size()*G->remainedRightVertices.size())/powl((1+epsilon),  sqrtl(logl(G->n)*delta));
            long double f2_treshold = sqrtl(G->remainedLeftVertices.size()*G->remainedRightVertices.size())/(candidate_z*powl((1+epsilon),  sqrtl(logl(G->n)*delta)));
            G->freeze_degree_gre(t1, t2);
            if (G->frozenLeftVertices.size() > f1_treshold || G->frozenRightVertices.size() > f2_treshold){
                return end_run_for_fixed_z_d();
            }

            for(int step_number = 0; step_number < T; step_number++){
                if( step_number && (step_number & (step_number - 1)) == 0){
                    number_of_rounds++;
                }
            
                long double s_t_portion = G->remainedLeftVertices.size()*1./G->remainedRightVertices.size();
                if(s_t_portion >= candidate_z*candidate_z){
                    G->get_left_degrees_in_sampled_graph(p1);
                    int number_of_left_vertices_degrees_less_than_p1k1 = 0;
                    for(int i = 0; i < G->remainedLeftVertices.size(); i++){
                        if(!G->remainedLeftVertices[i]->isFrozen && G->remainedLeftVertices[i]->sampledDegree < p1*k1){
                            number_of_left_vertices_degrees_less_than_p1k1++;
                        }
                    }
                    if(number_of_left_vertices_degrees_less_than_p1k1 < (epsilon/(1+epsilon))*G->remainedLeftVertices.size() - G->frozenLeftVertices.size()){
                        return end_run_for_fixed_z_d();
                    }
                    else{
                        finish_left_step();
                        if(number_of_left_vertices_degrees_less_than_p1k1 == 0){
                            break;
                        }
                    }
                }

                else{
                    G->get_right_degrees_in_sampled_graph(p2);
                    int number_of_right_vertices_degrees_less_than_p2k2 = 0;
                    for(int i = 0; i < G->remainedRightVertices.size(); i++){
                        if(!G->remainedRightVertices[i]->isFrozen && G->remainedRightVertices[i]->sampledDegree < p2*k2){
                            number_of_right_vertices_degrees_less_than_p2k2++;
                        }
                    }
                    if(number_of_right_vertices_degrees_less_than_p2k2 < (epsilon/(1+epsilon))*G->remainedRightVertices.size() - G->frozenRightVertices.size()){
                        return end_run_for_fixed_z_d();
                    }
                    else{
                        finish_right_step();
                        if(number_of_right_vertices_degrees_less_than_p2k2 == 0){
                            break;
                        }
                    }
                }
            }
            finish_phase();
        }
        return end_run_for_fixed_z_d();
    }


    void run(){
        long double min_candidate_d = (long double)G->edges.size()/(G->leftVertices.size()); 
        for(int i = 0; i < G->leftVertices.size(); i++){
            min_candidate_d = std::max(min_candidate_d, sqrtl(G->leftVertices[i]->edges.size()));
        }
        for(int i = 0; i < G->rightVertices.size(); i++){
            min_candidate_d = std::max(min_candidate_d, sqrtl(G->rightVertices[i]->edges.size()));
        }

        std::vector<long double> candidate_d_list = std::vector<long double>();
        std::vector<long double> candidate_z_list = std::vector<long double>();
        for(long double d = min_candidate_d; d <= G->n; d *= (1.L+epsilon)){
            candidate_d_list.push_back(d);
        }
        for(long double z = 1.L/sqrtl(G->n); z <= sqrtl(G->n); z *= (1.L+epsilon)){
            candidate_z_list.push_back(z);
        }

        std::string result_folder_path = "../../Results/";


        std::string epsilon_string =  std::to_string(epsilon);
        if(epsilon_string.find('.') != std::string::npos){
            while(epsilon_string[epsilon_string.size()-1] == '0'){
                epsilon_string = epsilon_string.substr(0, epsilon_string.size()-1);
                if(epsilon_string[epsilon_string.size()-1] == '.'){
                    epsilon_string = epsilon_string.substr(0, epsilon_string.size()-1);
                    break;
                }
            }
        }


        std::string delta_string =  std::to_string(delta);
        if(delta_string.find('.') != std::string::npos){
            while(delta_string[delta_string.size()-1] == '0'){
                delta_string = delta_string.substr(0, delta_string.size()-1);
                if(delta_string[delta_string.size()-1] == '.'){
                    delta_string = delta_string.substr(0, delta_string.size()-1);
                    break;
                }
            }
        }

        std::string result_file_name = algorithm_name + "___" + test_case + "___" + delta_string + "___" + epsilon_string + ".json";

        std::ifstream f(result_folder_path + result_file_name);
        if (f.good()) {
            std::cout << "File already exists. Do you want to clear it? (y/n): ";
            char choice;
            std::cin >> choice;
            if (choice == 'y') {
                std::ofstream ofs(result_folder_path + result_file_name, std::ofstream::trunc);
                ofs.close();
            } else {
                std::cout << "Exiting without clearing the file." << std::endl;
                return;
            }
        }
        f.close();


        std::ofstream resultFile(result_folder_path + result_file_name);
        if (!resultFile) {
            std::cerr << "Error occurred while writing to the file!" << std::endl;
        }        
        resultFile << "[";
        int number_of_candidates = candidate_d_list.size() * candidate_z_list.size();
        int counter = 0;
        for(auto z: candidate_z_list){
            for(auto d: candidate_d_list){
                counter++;
                std::cout << "Running for candidate_d: " << d << ", candidate_z: " << z << " (" << counter << "/" << number_of_candidates << ")" << std::endl;
                std::pair<int, long double> result = run_for_fixed_z_d(z, d);   
                if (resultFile.is_open()) {
                    resultFile << "{\n";
                    resultFile << "\"candidate_d\": " << d << ",\n";
                    resultFile << "\"candidate_z\": " << z << ",\n";
                    resultFile << "\"number_of_rounds\": " << result.first << ",\n";
                    resultFile << "\"density\": " << result.second << "\n";
                    resultFile << "},\n";
                    resultFile.flush();
                } else {
                    std::cerr << "Unable to open file" + result_folder_path + result_file_name + "\n";
                    exit(1);
                }
            }
        }
        resultFile.seekp(-2, std::ios_base::end);
        resultFile << "\n]";
        resultFile.flush();
        resultFile.close();
    }

    void reverse_run(){
        long double min_candidate_d = (long double)G->edges.size()/(G->leftVertices.size()); 
        for(int i = 0; i < G->leftVertices.size(); i++){
            min_candidate_d = std::max(min_candidate_d, sqrtl(G->leftVertices[i]->edges.size()));
        }
        for(int i = 0; i < G->rightVertices.size(); i++){
            min_candidate_d = std::max(min_candidate_d, sqrtl(G->rightVertices[i]->edges.size()));
        }

        std::vector<long double> candidate_d_list = std::vector<long double>();
        std::vector<long double> candidate_z_list = std::vector<long double>();
        for(long double d = min_candidate_d; d <= G->n; d *= (1.L+epsilon)){
            candidate_d_list.push_back(d);
        }
        for(long double z = 1.L/sqrtl(G->n); z <= sqrtl(G->n); z *= (1.L+epsilon)){
            candidate_z_list.push_back(z);
        }

        std::string result_folder_path = "../../Results/";


        std::string epsilon_string =  std::to_string(epsilon);
        if(epsilon_string.find('.') != std::string::npos){
            while(epsilon_string[epsilon_string.size()-1] == '0'){
                epsilon_string = epsilon_string.substr(0, epsilon_string.size()-1);
                if(epsilon_string[epsilon_string.size()-1] == '.'){
                    epsilon_string = epsilon_string.substr(0, epsilon_string.size()-1);
                    break;
                }
            }
        }


        std::string delta_string =  std::to_string(delta);
        if(delta_string.find('.') != std::string::npos){
            while(delta_string[delta_string.size()-1] == '0'){
                delta_string = delta_string.substr(0, delta_string.size()-1);
                if(delta_string[delta_string.size()-1] == '.'){
                    delta_string = delta_string.substr(0, delta_string.size()-1);
                    break;
                }
            }
        }

        std::string result_file_name = algorithm_name + "___" + test_case + "___" + delta_string + "___" + epsilon_string + ".json";

        std::ifstream f(result_folder_path + result_file_name);
        if (f.good()) {
            std::cout << "File already exists. Do you want to clear it? (y/n): ";
            char choice;
            std::cin >> choice;
            if (choice == 'y') {
                std::ofstream ofs(result_folder_path + result_file_name, std::ofstream::trunc);
                ofs.close();
            } else {
                std::cout << "Exiting without clearing the file." << std::endl;
                return;
            }
        }
        f.close();


        std::ofstream resultFile(result_folder_path + result_file_name);
        if (!resultFile) {
            std::cerr << "Error occurred while writing to the file!" << std::endl;
        }        
        resultFile << "[";
        int number_of_candidates = candidate_d_list.size() * candidate_z_list.size();
        int counter = 0;
        for(int i = candidate_z_list.size()-1; i > -1 ; i--){
            auto z = candidate_z_list[i];
            for(auto d: candidate_d_list){
                counter++;
                std::cout << "Running for candidate_d: " << d << ", candidate_z: " << z << " (" << counter << "/" << number_of_candidates << ")" << std::endl;
                std::pair<int, long double> result = run_for_fixed_z_d(z, d);   
                if (resultFile.is_open()) {
                    resultFile << "{\n";
                    resultFile << "\"candidate_d\": " << d << ",\n";
                    resultFile << "\"candidate_z\": " << z << ",\n";
                    resultFile << "\"number_of_rounds\": " << result.first << ",\n";
                    resultFile << "\"density\": " << result.second << "\n";
                    resultFile << "},\n";
                    resultFile.flush();
                } else {
                    std::cerr << "Unable to open file" + result_folder_path + result_file_name + "\n";
                    exit(1);
                }
            }
        }
        resultFile.seekp(-2, std::ios_base::end);
        resultFile << "\n]";
        resultFile.flush();
        resultFile.close();
    }

    void resume_running(long double last_z, long double last_d){
        long double min_candidate_d = (long double)G->edges.size()/(G->leftVertices.size()); 
        for(int i = 0; i < G->leftVertices.size(); i++){
            min_candidate_d = std::max(min_candidate_d, sqrtl(G->leftVertices[i]->edges.size()));
        }
        for(int i = 0; i < G->rightVertices.size(); i++){
            min_candidate_d = std::max(min_candidate_d, sqrtl(G->rightVertices[i]->edges.size()));
        }

        std::vector<long double> candidate_d_list = std::vector<long double>();
        std::vector<long double> candidate_z_list = std::vector<long double>();
        for(long double d = min_candidate_d; d <= G->n; d *= (1.L+epsilon)){
            candidate_d_list.push_back(d);
        }
        for(long double z = 1.L/sqrtl(G->n); z <= sqrtl(G->n); z *= (1.L+epsilon)){
            candidate_z_list.push_back(z);
        }


        std::string result_folder_path = "../../Results/";

        std::string epsilon_string =  std::to_string(epsilon);
        if(epsilon_string.find('.') != std::string::npos){
            while(epsilon_string[epsilon_string.size()-1] == '0'){
                epsilon_string = epsilon_string.substr(0, epsilon_string.size()-1);
                if(epsilon_string[epsilon_string.size()-1] == '.'){
                    epsilon_string = epsilon_string.substr(0, epsilon_string.size()-1);
                    break;
                }
            }
        }


        std::string delta_string =  std::to_string(delta);
        if(delta_string.find('.') != std::string::npos){
            while(delta_string[delta_string.size()-1] == '0'){
                delta_string = delta_string.substr(0, delta_string.size()-1);
                if(delta_string[delta_string.size()-1] == '.'){
                    delta_string = delta_string.substr(0, delta_string.size()-1);
                    break;
                }
            }
        }

        std::string result_file_name = algorithm_name + "___" + test_case + "___" + delta_string + "___" + epsilon_string + ".json";
        std::ofstream resultFile(result_folder_path + result_file_name, std::ios::app);

        int number_of_candidates = candidate_d_list.size() * candidate_z_list.size();
        int counter = 0;

        std::cout << last_z << ' ' << last_d << std::endl; 
        for(auto z: candidate_z_list){
            if(z < last_z){
                counter += candidate_d_list.size();
                continue; 
            }
            for(auto d: candidate_d_list){
                if(z == last_z && d <= last_d){
                    counter ++;
                    continue; 
                }
                counter++;
                std::cout << "Running for candidate_z: " << z << " ,candidate_d: " << d <<  " (" << counter << "/" << number_of_candidates << ")" << std::endl;
                std::pair<int, long double> result = run_for_fixed_z_d(z, d);   
                if (resultFile.is_open()) {
                    resultFile << "{\n";
                    resultFile << "\"candidate_d\": " << d << ",\n";
                    resultFile << "\"candidate_z\": " << z << ",\n";
                    resultFile << "\"number_of_rounds\": " << result.first << ",\n";
                    resultFile << "\"density\": " << result.second << "\n";
                    resultFile << "},\n";
                } else {
                    std::cerr << "Unable to open file" + result_folder_path + result_file_name + "\n";
                    exit(1);
                }
            }
        }
        resultFile.seekp(-2, std::ios_base::end);
        resultFile << "\n]";
        resultFile.flush();
        resultFile.close();
    }
};

void graph_remaining_info_v1(GraphV1* G){
    G->initialize_remained_graph();
    std::cout << "Left vertices: " << G->remainedLeftVertices.size() << std::endl;
    for(int i = 0; i < G->remainedLeftVertices.size(); i++){
        std::cout << "Left vertex " << i << ": ";
        for(int j = 0; j < G->remainedLeftVertices[i]->edges.size(); j++){
            std::cout << G->remainedLeftVertices[i]->edges[j]->v2->id << " ";
        }
        std::cout << std::endl;
    }
    std::cout << "Right vertices: " << G->remainedRightVertices.size() << std::endl;
    for(int i = 0; i < G->remainedRightVertices.size(); i++){
        std::cout << "Right vertex " << i << ": ";
        for(int j = 0; j < G->remainedRightVertices[i]->edges.size(); j++){
            std::cout << G->remainedRightVertices[i]->edges[j]->v1->id << " ";
        }
        std::cout << std::endl;
    }
    std::cout << "Edges: " << G->remainedEdges.size() << std::endl;
    for(int i = 0; i < G->remainedEdges.size(); i++){
        std::cout << "Edge " << i << ": " << G->remainedEdges[i]->v1->id << " " << G->remainedEdges[i]->v2->id << " " << G->remainedEdges[i]->edgeListId <<  " " << G->remainedEdges[i]->leftId <<  " " << G->remainedEdges[i]->rightId << std::endl;
    }
}

void graph_frozen_info_v1(GraphV1* G){
    std::cout << "Frozen left vertices: ";
    for(int i = 0; i < G->frozenLeftVertices.size(); i++){
        std::cout << G->frozenLeftVertices[i]->id << " ";
    }
    std::cout << std::endl;
    std::cout << "Frozen right vertices: " ;
    for(int i = 0; i < G->frozenRightVertices.size(); i++){
        std::cout << G->frozenRightVertices[i]->id << " ";
    }
    std::cout << std::endl;
}

std::pair<int, int>* n_m(std::vector<std::pair<int, int>*> edges){
    std::pair<int, int>* n_m = new std::pair<int, int>(0, edges.size());
    for(int i = 0; i < edges.size(); i++){
        n_m->first = std::max(n_m->first, std::max(edges[i]->first, edges[i]->second));
    }
    return n_m;
}

std::vector<std::pair<int, int>*> get_edges(std::string file_path){
    std::vector<std::pair<int, int>*> edges = std::vector<std::pair<int, int>*>();
    std::ifstream file(file_path);

    if (!file.is_open()) {
        std::cerr << "Error: Could not open file " << file_path << std::endl;
        return edges;
    }

    std::string line;
    while (std::getline(file, line)) {
        std::istringstream iss(line);
        int a, b;

        if (iss >> a >> b) {
            edges.push_back(new std::pair<int, int>(a, b));
        } else {
            std::cerr << "Warning: Skipping malformed line: " << line << std::endl;
        }
    }
    file.close();
    return edges;
}

int main(){
    auto edges = get_edges("../../Datasets/Slashdot0902.txt");   
    GraphV1* G = new GraphV1(edges, n_m(edges)->first + 1);
    OurMPCAlgorithm* alg = new OurMPCAlgorithm(G, 0.8, 0.2, "Slashdot0902");
    alg->reverse_run(); 
}